﻿# PowerShell MPatch Installer
# 
# Version 2.3	Mar 2024
# For Patching Ev6
# verify that the file c:\windows\syswow64\printdisp.exe exists
# Add log
# update how the config files are updated.    
# 
#If Unsigned PS's are boloced, update the path where you have the installer locaated and run this line
# unblock-file -path c:\temp\*.ps1
<#
Set-ExecutionPolicy -ExecutionPolicy unrestricted -Scope process # not digitally signed
Set-ExecutionPolicy -ExecutionPolicy unrestricted -Scope LocalMachine # not digitally signed
Set-ExecutionPolicy unrestricted -Scope CurrentUser
#>

# Check that version is legal for this software
# https://en.wikipedia.org/wiki/List_of_Microsoft_Windows_versions

$OSVersion = [System.Environment]::OSVersion.Version.build

if ($OSVersion -lt 9200 ) { # less than windows 2012.
    read-host -Prompt "Minimum requirements for this software is Windows Server 2012"
    exit
  }
  elseif ($OSVersion -lt 14393) { # less than windows 2016
    write-host "Warning Windows 2012 is scheduled for EOL as of Oct 2023"
}

    

Import-Module WebAdministration

if (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    Write-Warning 'You do not have Administrator rights to run this script!  Please re-run this script as an Administrator!' -WarningAction stop
}
else {
    $runas = 'Script was run as Administrator'
}

#Is the Faxcore eV6 Software installed?

if (-NOT (Test-Path 'HKLM:SYSTEM\CurrentControlSet\Services\FXC6.ServicesAPI')){
    return 'No Faxcore eV6 Software detected!  ABORTING'
    }

#Get location of mpatch
$BaseInstallFromDir = $MyInvocation.MyCommand.Path
if (!$BaseInstallFromDir) {
    $BaseInstallFromDir = $psISE.CurrentFile.Fullpath
}
if ($BaseInstallFromDir) {
    $BaseInstallFromDir = Split-Path $BaseInstallFromDir -Parent
}
# Eliminate the prompting to run
$logPath =  $BaseInstallFromDir + '\Mpatch-' + (Get-Date).ToString("yyyyMMdd-HHmmss")+'.log'
Start-Transcript -Path $logPath -Append -IncludeInvocationHeader -Force

Get-ChildItem -Path $BaseInstallFromDir -Recurse | Unblock-File

#Functions

function StopService{
    param($ServiceName)#Functions
    write-output $ServiceName
    $ServiceNameExists = Get-Service -Name $ServiceName -ErrorAction SilentlyContinue   # Does the service exist?
    if($ServiceNameExists -ne $null) {
        $ServiceInfo = Get-WmiObject win32_service | ?{$_.Name -eq $ServiceName}  # get the info about the service before trying to stop the service and loose that info.
        $FAPid = $serviceinfo.processid # preservice the PID.
        $arrService = Get-Service -Name $ServiceName # what is the service doing
        if ($arrService.Status -ne 'stopped'){   # is the service running
            Stop-Service $ServiceName -Force -ErrorAction SilentlyContinue # stop the service
            Start-Sleep -s 5
        }
        if ($serviceinfo.state -eq 'running') { # Is the service running
            write-output "Stopping $ServiceName"
            Stop-process -id $FAPid -force -ErrorAction SilentlyContinue  # stop the process, and that will stop the service
        } 
    }
} # StopService

function UpdateConfigFiles{
    param([string]$ProdConfigPath, [string]$ConfigID) #Functions
        write-output "Config File Merge for $ProdConfigPath\$ConfigID"
        $ConfigIDBak = $ConfigID + "_mpatch_bak"

        #does the folder exist
        if (Test-Path -Path $ProdConfigPath) {

            #does the config file exist
            if (Test-Path -Path $ProdConfigPath/$ConfigID -PathType leaf) {

                # Backup the production .config file
                copy-item "$ProdConfigPath\$ConfigID" -Destination "$ProdConfigPath\$ConfigIDBak" -Force  # save the origial pre-formatted
                # Format (print-pritty) the production file
                start-process -wait -filepath "$BaseInstallFromDir\tools\tidy.exe" -argumentList " -xml -o $ProdConfigPath\$ConfigID -i $ProdConfigPath\$ConfigIDBak "  # format/print-pretty the production config file
                Remove-item -Path $ProdConfigPath\$ConfigIDBak # clean up redundant file

                # Backup the default .config file
                copy-item "$ProdConfigPath\DefaultConfigs\$ConfigID" -Destination "$ProdConfigPath\DefaultConfigs\$ConfigIDBak" -Force  # save the Default config pre-formatted
                # Format (print-pritty) the default file
                start-process -wait -filepath "$BaseInstallFromDir\tools\tidy.exe" -argumentList " -xml -o $ProdConfigPath\DefaultConfigs\$ConfigID -i $ProdConfigPath\DefaultConfigs\$ConfigIDBak " # format/print-pretty the Defaul config file
                Remove-item -Path $ProdConfigPath\DefaultConfigs\$ConfigIDBak  # clean up redundant file

                $file1 = "$ProdConfigPath\$ConfigID"
                $file2 = "$ProdConfigPath\DefaultConfigs\$ConfigID"
                if(Compare-Object -ReferenceObject $(Get-Content $file1) -DifferenceObject $(Get-Content $file2)) { # only force them to look at files that differ.

                    # compare/merge the default .config into the production .config
                    start-process -wait -filepath "$BaseInstallFromDir\Tools\WinMergeU.exe" -argumentList "$ProdConfigPath\DefaultConfigs\$ConfigID $ProdConfigPath\$ConfigID"

                } 
  
            } else {

                "$ProdConfigPath/$ConfigID does not exist"

            }

        } else {

            "$ProdConfigPath does not exist"
        }

} # end of UpdateConfigFiles

# end of function defs

#Where is faxcore Installed
#look in the registry for where the dispatcher service starts.
$FCPath = split-path(split-path (Get-ItemProperty "Registry::HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\FXC6.ServicesAPI" -Name "ImagePath").ImagePath)
# check if the registry has quotes around the path
if ($FCPath.substring(0,1) -eq '"') {
	$FCPath = $FCPath.substring(1, $FCPath.Length-1)
}
#
$title = "Site to Update"
$message = "What Site would you like to update?"
$WebSites = Get-ChildItem -Path IIS:\Sites -Exclude 'Default Web Site'
If ($WebSites.count -eq 1) {    # If there is only one website, assume that is what we are going to update
    $iisWebsiteName = $WebSites.name
} else {
    $options = [System.Management.Automation.Host.ChoiceDescription[]] @($WebSites.name)
    [int]$defaultchoice = 0
    $opt = $host.UI.PromptForChoice($Title , $message, $Options,$defaultchoice)
    [string] $iisWebsiteName = $WebSites.name[$opt]
}
#
#[string] $iisWebsiteName = 'Faxcore'
#$FCPathWeb = Get-WebFilePath "IIS:\Sites\$iisWebsiteName" # get the web location based on where IIS says it is
$FCPathWeb = ((Get-IISSite "$iisWebsiteName").Applications.VirtualDirectories.PhysicalPath)  # get the web location based on where IIS says it is
if ($FCPathWeb -is [array]) {  # check if it returns only the site or an array of sites
    $FCPathWeb = $FCPathWeb[0] 
}

#update permissions so this process can update \faxcore

#
# Get the ACL for an existing folder
$existingAcl = Get-Acl -Path "$FCPath"

# Set the permissions that you want to apply to the folder
$permissions = $env:username, 'FullControl', 'ContainerInherit,ObjectInherit', 'None', 'Allow'

# Create a new FileSystemAccessRule object
$rule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList $permissions

# Modify the existing ACL to include the new rule
$existingAcl.SetAccessRule($rule)

# Apply the modified access rule to the folder(s)
  write-host 'Updating permissions'
  $existingAcl | Set-Acl -Path "$FCPath\svc.Dispatcher\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\svc.Dispatcher\*.exe"
  $existingAcl | Set-Acl -Path "$FCPath\svc.Faxagent\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\svc.Faxagent\*.exe"
  $existingAcl | Set-Acl -Path "$FCPath\svc.FileGateway\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\svc.FileGateway\*.exe"
  $existingAcl | Set-Acl -Path "$FCPath\svc.InternalAPI\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\svc.InternalAPI\*.exe"
  $existingAcl | Set-Acl -Path "$FCPath\svc.RenderAgent\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\svc.RenderAgent\*.exe"
  $existingAcl | Set-Acl -Path "$FCPath\svc.Report\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\svc.Report\*.exe"
  $existingAcl | Set-Acl -Path "$FCPath\svc.SMTPGateway\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\svc.SMTPGateway\*.exe"
  $existingAcl | Set-Acl -Path "$FCPath\web\*.dll"
  $existingAcl | Set-Acl -Path "$FCPath\web\*.exe"
#
<#
    Write-Host "Double check permissions"
    $acl = Get-Acl -Path $FCPath
    $permission = 'IIS_IUSRS', 'Modify', 'ContainerInherit, ObjectInherit', 'None', 'Allow'
    $rule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList $permission
    $acl.SetAccessRule($rule)
    $acl | Set-Acl -Path $FCPath

    $acl = Get-Acl -Path $FCPathWeb
    $permission = 'IIS_IUSRS', 'Modify', 'ContainerInherit, ObjectInherit', 'None', 'Allow'
    $rule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList $permission
    $acl.SetAccessRule($rule)
    $acl | Set-Acl -Path $FCPathWeb
    
    # Give everyone permissions to update the faxcore folder.
    $acl = Get-Acl -Path $FCPath
    $temppermission = 'Everyone', 'FullControl', 'ContainerInherit, ObjectInherit', 'None', 'Allow'
    $temprule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList $temppermission
    $acl.SetAccessRule($temprule)
    $acl | Set-Acl -Path $FCPath
#>
#Create backup folders structure 
    $BackupPath = $FCPath + '\update'
    Write-Host $BackupPath
    if(!(Test-Path -Path $BackupPath )){
        New-Item -ItemType directory -Path $BackupPath
        Write-Host "New folder created"
    }
    #Add unique folder for this run
    $BackupPath = $BackupPath + '\Mpatch-' + (Get-Date).ToString("yyyyMMdd-HHmmss")
    Write-Host $BackupPath
    New-Item -ItemType directory -Path $BackupPath
        New-Item -ItemType directory -Path "$BackupPath\SVC\svc.dispatcher"
        New-Item -ItemType directory -Path "$BackupPath\SVC\svc.faxagent"
        New-Item -ItemType directory -Path "$BackupPath\SVC\svc.filegateway"
        New-Item -ItemType directory -Path "$BackupPath\SVC\svc.internalapi"
        New-Item -ItemType directory -Path "$BackupPath\SVC\svc.renderagent"
        New-Item -ItemType directory -Path "$BackupPath\SVC\svc.report"
        New-Item -ItemType directory -Path "$BackupPath\SVC\svc.smtpgateway"
        New-Item -ItemType directory -Path "$BackupPath\Web"
        #Branding Folders
        New-Item -ItemType directory -Path "$BackupPath\Web\Content"
        New-Item -ItemType directory -path "$BackupPath\Web\Content\img"
        New-Item -ItemType directory -Path "$BackupPath\Web\Content\icons"
        New-Item -ItemType directory -Path "$BackupPath\Web\Content\material-pro"
        New-Item -ItemType directory -Path "$BackupPath\Web\Content\material-pro\css"
        New-Item -ItemType directory -Path "$BackupPath\Web\Content\material-pro\css\colors"
        New-Item -Itemtype directory -Path "$BackupPath\Web\Common"
        New-Item -Itemtype directory -Path "$BackupPath\Web\Common\Templates"
        New-Item -ItemType directory -Path "$BackupPath\Web\Areas\"
        New-Item -ItemType directory -Path "$BackupPath\Web\Areas\Admin"
        New-Item -ItemType directory -Path "$BackupPath\Web\Areas\Admin\Views"
        New-Item -ItemType directory -Path "$BackupPath\Web\Areas\Admin\Views\Support"

# backup config files and images
    #backup svc configs
    Write-Host "backup the \\faxcore configs"
    Copy-Item "$FCPath\svc.dispatcher\*.config" -Destination "$BackupPath\SVC\svc.dispatcher"
    Copy-Item "$FCPath\svc.faxagent\*.config" -Destination "$BackupPath\SVC\svc.faxagent"
    Copy-Item "$FCPath\svc.filegateway\*.config" -Destination "$BackupPath\SVC\svc.filegateway"
    Copy-Item "$FCPath\svc.internalapi\*.config" -Destination "$BackupPath\SVC\svc.internalapi"
    Copy-Item "$FCPath\svc.renderagent\*.config" -Destination "$BackupPath\SVC\svc.renderagent"
    Copy-Item "$FCPath\svc.report\*.config" -Destination "$BackupPath\SVC\svc.report"
    Copy-Item "$FCPath\svc.smtpgateway\*.config" -Destination "$BackupPath\SVC\svc.smtpgateway"
    #backup web configs
    Write-Host "backup the \\faxcore web configs"
    Copy-Item "$FCPathWeb\*.config" -Destination "$BackupPath\WEB"

    #backup branding images
    Write-Host "backup the branding images"
    If (Test-Path -Path "$FCPathWeb\Content\img\BG-login.jpg" ) {
        Copy-Item "$FCPathWeb\Content\img\BG-login.jpg" -Destination "$BackupPath\WEB\Content\img" -Force 
    }
    If (Test-Path -Path "$FCPathWeb\Content\img\login_logo.png" ) {
        Copy-Item "$FCPathWeb\Content\img\login_logo.png" -Destination "$BackupPath\WEB\Content\img" -Force 
    }
    If (Test-Path -Path "$FCPathWeb\Content\img\logo.png" ) {
        Copy-Item "$FCPathWeb\Content\img\logo.png" -Destination "$BackupPath\WEB\Content\img"-Force 
    }
    
    # Common\Templates
    If (Test-Path -Path "$FCPathWeb\Common\Templates") {
        Copy-Item "$FCPathWeb\Common\Templates\eMail*.txt" -Destination "$BackupPath\WEB\Common\Templates" -Force
    }
    
    # Content\icons
    If (Test-Path -Path "$FCPathWeb\Content\icons") {
        Copy-Item "$FCPathWeb\Content\icons\favicon*.png" -Destination "$BackupPath\WEB\Content\icons" -Force
    }

    # Content\ colors
    If (Test-Path -Path "$FCPathWeb\Content\material-pro\css\colors\blue-dark.css") {
        Copy-Item "$FCPathWeb\Content\material-pro\css\colors\blue-dark.css" -Destination "$BackupPath\WEB\Content\material-pro\css\colors" -Force
    }

    # support verbage
    If (Test-Path -Path "$FCPathWeb\Areas\Admin\Views\Support\Help.cshtml") {
        Copy-Item "$FCPathWeb\Areas\Admin\Views\Support\Help.cshtml" -Destination "$BackupPath\WEB\Areas\Admin\Views\Support\Help.cshtml" -Force
    }

    #backup Brooktrout, if installed
    if((Test-Path -Path "C:\Program Files (x86)\Brooktrout\config")){
        Write-Host "backup the brooktrout config"
        New-Item -ItemType directory -Path "$BackupPath\Brooktrout"
        Copy-Item "C:\Program Files (x86)\Brooktrout\config\*.cfg" -Destination "$BackupPath\Brooktrout"
    }


#copy Mpatch Change Logs to backup file
if((Test-Path -Path "$BaseInstallFromDir\ChangeLog.txt")){
    Copy-Item "$BaseInstallFromDir\ChangeLog.txt" -Destination $BackupPath
}

#Stop services
write-host "stopping the sevices"
StopService "fxc6.faxagent"
StopService "fxc6.dispatchagent"
StopService "fxc6.renderagent"
StopService "fxc6.filegateway"
StopService "fxc6.reportAPI"
StopService "fxc6.servicesAPI"
StopService "fxc6.smtpgatewayagent"
StopService "SQL Server (FXCDB)"

#did they really stop?
    $Services = Get-WmiObject -Class win32_service -Filter "state = 'stop pending'"
    if ($Services) {
        foreach ($service in $Services) {
        try {
                Stop-Process -Id $service.processid -Force -PassThru -ErrorAction Stop
            }
            catch {
                Write-Warning -Message " Error. Error details: $_.Exception.Message"
            }
        }
    }

# Up the service start timeout
$test = test-path -path "HKLM:\SYSTEM\CurrentControlSet\Controls\ServicesPipeTimeout"

if(-not($test)){
    $registryPath = "HKLM:\SYSTEM\CurrentControlSet\Control"
    $Name = "ServicesPipeTimeout"
    $value = "240000"
    New-ItemProperty -Path $registryPath -Name $name -Value $value  -PropertyType DWORD -Force | Out-Null
}
#Stop IIS
Write-Host "Stop IIS"
invoke-command -scriptblock {iisreset /STOP}

Start-Sleep -s 15
Write-Host "IIS Stopped"

#update software

$ErrorActionPreference = 'Stop'

    # Use the same ev6 that the installer uses

    # orig Expand-Archive -path "$BaseInstallFromDir\FAXCORE_EV6.ZIP" -DestinationPath $FCPath -force
    write-host "unzip FaxCoreWeb.zip"
    Expand-Archive -path "$BaseInstallFromDir\FaxCoreCode\FaxCoreWeb.zip" -DestinationPath $FCPathWeb -force -Verbose   #unzip the web folder to where IIS says the web is for that site
    Write-host "$FCPathWeb\papertown.txt" # Update audit trail
    Get-Content "$FCPathWeb\papertown.txt" # update audit trail
    Get-ChildItem -Path "$FCPathWeb\papertown.txt" # update audit trail
    write-host "unzip FaxCoreSrc.zip"
    Expand-Archive -path "$BaseInstallFromDir\FaxCoreCode\FaxCoreSrc.zip" -DestinationPath $FCPath -force -Verbose #unzip the faxcore folder to where the internalAPI's are located
    Write-host "$FCPath\papertown.txt" # Update audit trail
    Get-Content "$FCPath\papertown.txt" # Update audit trail
    Get-ChildItem -Path "$FCPath\papertown.txt" # update audit trail
    write-host "Faxcore software unzipped/installed"

    # pin the faxcore folder to the quick access
    $quickaccess = new-object -com shell.application
    $quickaccess.Namespace($FCPath).Self.InvokeVerb("pintohome")

# Update config files


# Update the web
$ProdConfigPath = "$FCPath\web"
$ConfigID = 'web.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of web

# Update the Dispatcher
$ProdConfigPath = "$FCPath\svc.Dispatcher"
$ConfigID = 'DispatchAgent.exe.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of Dispatcher

# Update the Faxagent
$ProdConfigPath = "$FCPath\svc.Faxagent"
$ConfigID = 'Faxagent.exe.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of faxagent

# Update the FileGateway
$ProdConfigPath = "$FCPath\svc.FileGateway"
$ConfigID = 'FileGatewayAgent.exe.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of FileGateway

# Update the FaxCore.Internal.API.exe.config
$ProdConfigPath = "$FCPath\svc.internalapi"
$ConfigID = 'FaxCore.Internal.API.exe.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of FaxCore.Interal.API.Config

# Update the RenderAgent
$ProdConfigPath = "$FCPath\svc.RenderAgent"
$ConfigID = 'RenderAgent.exe.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of RenderAgent

# Update the Report
$ProdConfigPath = "$FCPath\svc.Report"
$ConfigID = 'FaxCore.Report.API.exe.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of RenderAgent

# Update the SMTPGateway
$ProdConfigPath = "$FCPath\svc.SMTPGateway"
$ConfigID = 'SMTPGateway.exe.config'
UpdateConfigFiles $ProdConfigPath $ConfigID
# end of SMTPGateway

# end Update config files

#start the SQL
    
    #
    Write-Host "Is SQL Local?"
    $arrService = Get-Service -Name "SQL Server (FXCDB)" -ErrorAction SilentlyContinue   # is SQL running on this server?
    if($arrservice -ne $null)
    {
        # Service does exist
        if ($arrservice.StartType -eq 'Automatic') {
            Write-Host "Start SQL"
            Start-Service "SQL Server (FXCDB)"   # if service is present, start SQL
        }

    }
# Update the config files

	# update config files

# Restore Branding
Write-Host "Restore the branding images" 
If (Test-Path -Path "$BackupPath\WEB\Content\img\BG-login.jpg" ) {
    Copy-Item "$BackupPath\WEB\Content\img\BG-login.jpg" -Destination "$FCPathWeb\Content\img" -Force 
}
If (Test-Path -Path "$BackupPath\Web\Content\img\login_logo.png" ) {
    Copy-Item "$BackupPath\Web\Content\img\login_logo.png" -Destination "$FCPathWeb\Content\img" -Force 
}
If (Test-Path -Path "$BackupPath\Web\Content\img\logo.png" ) {
    Copy-Item "$BackupPath\Web\Content\img\logo.png" -Destination "$FCPathWeb\Content\img" -Force 
}

if (Test-Path -Path "$BackupPath\WEB\Common\Templates") {
    Copy-Item "$BackupPath\WEB\Common\Templates\*.*" -Destination "$FCPathWeb\Common\Templates" -Force
}

if (Test-Path -Path "$BackupPath\Web\Content\icons") {
    Copy-Item "$BackupPath\Web\Content\icons\*.png" -Destination "$FCPathWeb\Content\icons" -Force
}

# Content\ colors
If (Test-Path -Path "$BackupPath\Web\Content\material-pro\css\colors\blue-dark.css") {
    Copy-Item "$BackupPath\WEB\Content\material-pro\css\colors\blue-dark.css" -Destination "$FCPathWeb\Content\material-pro\css\colors" -Force
}

    # support verbage
If (Test-Path -Path "$BackupPath\WEB\Areas\Admin\Views\Support\Help.cshtml") {
    Copy-Item "$BackupPath\WEB\Areas\Admin\Views\Support\Help.cshtml" -Destination "$FCPathWeb\Areas\Admin\Views\Support\Help.cshtml" -Force
}

#Copy required Windows exe/dll's 

    if ((test-path -path c:\windows\syswow64\printdisp.exe -PathType leaf) -eq 0) {
        # Needed to run faxcore rendering
        copy-item c:\windows\system32\printdisp.exe -Destination c:\windows\syswow64 -Force
   }

# start IIS FaxCore.Internal.API.exe.config

Write-Host "Start IIS"
invoke-command -scriptblock {iisreset /Start}

Start-Sleep -s 5
#open the localhost web page to have it build the SQL DB
Write-Host " Start the web UI to kick off the SQL DB updater"
Start-Process http://localhost/admin
$(Write-Host "`n`n`n`n`n`n`n`n`n`nPlease press any key AFTER the web UI displays the login prompt`n`n`n`n`n`n`n`n" -ForegroundColor white -BackgroundColor red -NoNewline; Read-Host)


Write-Host " Running ServicesRestartEv6.bat"
powershell -command "Start-Process -FilePath $BaseInstallFromDir\ServicesRestartEv6.bat -Verb runas"
Stop-Transcript
Copy-Item $logPath -Destination "$BackupPath"

# SIG # Begin signature block
# MIIFeQYJKoZIhvcNAQcCoIIFajCCBWYCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUCEqjtahA8T1SS7g6wIPVtTFC
# XFSgggMQMIIDDDCCAfSgAwIBAgIQXxPQdh8WYqVH6qmqZJbQhDANBgkqhkiG9w0B
# AQsFADAeMRwwGgYDVQQDDBNGYXhjb3JlRXY2SW5zdGFsbGVyMB4XDTIxMDMxOTE5
# MjQxOFoXDTIyMDMxOTE5NDQxOFowHjEcMBoGA1UEAwwTRmF4Y29yZUV2Nkluc3Rh
# bGxlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMraPL+L5THZHaxk
# U1Op4Eok4YX5YmtuT9RAdzAGP2P88w0PeNl9qzqywRsKytkvtR4ijj5Ho2hM1LOU
# aQjL4gqD+C6G7TdheBE/ZpRK/Zm3AKE8iv198Mp4thmpT1hO3Tlb0Pa4PTz52sXN
# DPOBGeuVdU0nBvO/ARSnORQiFzouAkIA5MB70au/sw3TN8t+IfbQ0xt5Oe0pVI05
# CN/gJTCnETDCCOvb6PN62WAmjD+Sfrb8K8daORu009TDoZ0qb1v5yVTt11xyv7lI
# xgSqx2wrQe8huHzOF7A7Zl+v2ihoQIum7GMNacjHi51Np9Es5K3yFyLl88cUrqWG
# fpRjTwECAwEAAaNGMEQwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMB0GA1UdDgQWBBROayTTphtBFnfsXDA9busfvWi5hDANBgkqhkiG9w0BAQsF
# AAOCAQEADXL4/+Vv831jSJrnrS3H0sRfdeb1l43/IV9BePklLyW+NO5iXfZz6t79
# /Y6N7sQUXHTf9ADY+B87USc0iPltSPBMLeBR78EIJM9iNwYfl+r9B2lA/hE/Iqqh
# ycAezB0Lcmyrsc2YRgDyuPKTT3NyarEtV5y9cgkRg5XoTIoEudveK/+FvQq0Iyfi
# fj5WxjYtT894AW1Iu4u5sOQgrwrdaSxTlQTZWeqEDe42d0r7LnqCcjAOZ56V8o6W
# RUvqySwj+GAm+mBqmbqbhhYfQZOoxwaSVE/kikyDBH4b6Aj/HlxzYPS7lemNqOw5
# pnpwwvC6UBeWB3RGkrqYfYyOiVwZPzGCAdMwggHPAgEBMDIwHjEcMBoGA1UEAwwT
# RmF4Y29yZUV2Nkluc3RhbGxlcgIQXxPQdh8WYqVH6qmqZJbQhDAJBgUrDgMCGgUA
# oHgwGAYKKwYBBAGCNwIBDDEKMAigAoAAoQKAADAZBgkqhkiG9w0BCQMxDAYKKwYB
# BAGCNwIBBDAcBgorBgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAjBgkqhkiG9w0B
# CQQxFgQUJeshCigl9qwglRRSJfL8eZ/T704wDQYJKoZIhvcNAQEBBQAEggEArN2A
# 0UlIBKLNwpC8xgM8hzIrlksDejzXtKjjinljmHeaAkL9rkf2GIOpqlGmUi2mJ7m+
# Ka94ldfVlB3SfDFsWKOF7uklbe85PwjQeF7D31Xk1jO431e/tAn75SmKyZfwqRN/
# czRNNsG8mBcqPEraTEodbA0aqmN8+BONrQlVa18y0CojB2hAvb7Ssv3jQRPqrqKZ
# LAgJS8bmCWkzLUOIfcdTAwE1CBcL/tQm59e+Ta2HTVA2aBIQspjJjANeBLLzHa+K
# jVQeUIdmFO5DiorkX/LykYxl2rUU1i+NKMmmDl49D+PofWfDPAWVupIO5nGWvbOL
# nUn3YgAvwfbBQwvdTA==
# SIG # End signature block
