﻿#Requires -Version 5.1
#Requires -RunAsAdministrator

<#
.SYNOPSIS
    FaxCore eV6 MPatch Installer v3.1
.DESCRIPTION
    Safely updates FaxCore eV6 services and web components with config merging,
    branding preservation, and rollback protection.
#>

Set-StrictMode -Version Latest
$ErrorActionPreference = 'Stop'

# ===================================
# Load IIS Module Safely
# ===================================
if (-not (Get-Module -ListAvailable WebAdministration)) {
    Write-Error "WebAdministration module not found. Install IIS Management Scripts and Tools."
}
Import-Module WebAdministration

# ===================================
# Paths & Logging
# ===================================
# Determine base directory of this script
$BaseReportDir = $MyInvocation.MyCommand.Path
if (!$BaseReportDir) {
    $BaseReportDir = $psISE.CurrentFile.Fullpath
}
if ($BaseReportDir) {
    $BaseReportDir = Split-Path $BaseReportDir -Parent
}

$LogFile = Join-Path $BaseReportDir ("Mpatch-{0:yyyyMMdd-HHmmss}.log" -f (Get-Date))
Start-Transcript -Path $LogFile -IncludeInvocationHeader -Force

Write-Host "=== FaxCore eV6 MPatch Installer v3.1 ===" -ForegroundColor Cyan
Get-ChildItem -Path $BaseReportDir -Recurse | Unblock-File

# ===================================
# Pre-flight Checks
# ===================================
$OSBuild = [Environment]::OSVersion.Version.Build
if ($OSBuild -lt 9200) {
    Write-Error "Unsupported OS build: $OSBuild"
}

$svcReg = 'HKLM:\SYSTEM\CurrentControlSet\Services\FXC6.ServicesAPI'
if (-not (Test-Path $svcReg)) {
    Write-Error "FaxCore eV6 not detected."
}

$FCExe  = (Get-ItemProperty $svcReg).ImagePath -replace '^"|"$'
$FCPath = Split-Path (Split-Path $FCExe)
if (-not (Test-Path $FCPath)) {
    Write-Error "Invalid FaxCore path: $FCPath"
}

# ===================================
# IIS Site Selection
# ===================================
$WebSites  = @(Get-ChildItem IIS:\Sites | Where-Object Name -ne 'Default Web Site')
$siteCount = $WebSites.Count

switch ($siteCount) {
    0 { Write-Error "No non-default IIS sites found." }
    1 { 
        $Site = $WebSites[0]
        $PatchType = 'Base'  # default to updating all of faxcore
      }
    Default {
        # Multiple web sites.  Ask user for the one to work with
        $choices = $WebSites.Name | ForEach-Object {
            New-Object System.Management.Automation.Host.ChoiceDescription $_, $_
        }
        $idx  = $Host.UI.PromptForChoice("Select Site","Which IIS Web Site folder do you want to update",$choices,0)
        $Site = $WebSites[$idx]

        # ask if we should update just the web folder for this site or all the faxcore folders
        
        $patchChoices = @(
            New-Object System.Management.Automation.Host.ChoiceDescription "&Base MPatch", "Apply Base MPatch"
            New-Object System.Management.Automation.Host.ChoiceDescription "&Web Patch",  "Apply Web-only patch"
        )
        $patchIdx = $Host.UI.PromptForChoice(
            "Patch Type",
            "Which type of patch do you want to apply to '$($Site.Name)'?",
            $patchChoices,
            0   # Default to Base patch
        )
        $PatchType = switch ($patchIdx) {
            0 { "Base" }
            1 { "Web" }
        }
    }
}

$iisWebsiteName = $Site.Name
$FCPathWeb = [Environment]::ExpandEnvironmentVariables($Site.PhysicalPath)
$webFolder = (Split-Path $FCPathWeb -Leaf).TrimEnd('\')
Write-Host "Updating site: $iisWebsiteName" -ForegroundColor Green
Write-Host "Web root: $FCPathWeb"
Write-Host "Service root: $FCPath"
Write-Host "Updating all or just web site: $PatchType"

# ===================================
# Backup Structure
# ===================================
Write-Host "Backing up important files"

$BackupRoot = Join-Path "$FCPath\update" ("Mpatch-{0:yyyyMMdd-HHmmss}" -f (Get-Date))
if (-not (Test-Path $BackupRoot)) { New-Item -ItemType Directory -Path $BackupRoot -Force | Out-Null }

# Backup service .config files, but only if its a base mpatch 
if ($PatchType -eq 'Base') { 
    $svcBkupFolders = Get-ChildItem -Path $FCPath -Directory -Filter 'svc.*'
    foreach ($svcFolder in $svcBkupFolders) {
        $folderName = $svcFolder.Name
        $folderPath = $svcFolder.FullName

        $destFolder = Join-Path $BackupRoot $folderName
        if (-not (Test-Path $destFolder)) { New-Item -ItemType Directory -Path $destFolder -Force | Out-Null }

        Write-Host "   Backing up .config files in $folderName" -ForegroundColor Green
        Get-ChildItem -Path "$folderPath\*.config" -File -ErrorAction SilentlyContinue | ForEach-Object {
            Copy-Item $_.FullName -Destination $destFolder -Force
        }
    }
}

# Backup web .config files
$destWebFolder = Join-Path $BackupRoot $webFolder
if (-not (Test-Path $destWebFolder)) { New-Item -ItemType Directory -Path $destWebFolder -Force | Out-Null }

Write-Host "   Backing up .config files from $FCPathWeb\*.config" -ForegroundColor Green
Get-ChildItem -Path "$FCPathWeb\*.config" -File -ErrorAction SilentlyContinue | ForEach-Object {
    Copy-Item $_.FullName -Destination $destWebFolder -Force
}

# ===================================
# Backup Branding files
# ===================================

Write-Host "Backup Branding"
$brandingFiles = @(
    "Content\img\BG-login.jpg",
    "Content\img\login_logo.png",
    "Content\img\logo.png",
    "Content\icons\favicon*.png",
    "Content\material-pro\css\colors\blue-dark.css",
    "Common\Templates\eMail*.txt",
    "Areas\Admin\Views\Support\Help.cshtml"
)

foreach ($file in $brandingFiles) {

    $prodPattern   = Join-Path $FCPathWeb $file
    $backupFolder = Join-Path (Join-Path $BackupRoot $webFolder) ([IO.Path]::GetDirectoryName($file))


    if (-not (Test-Path $backupFolder)) {
        New-Item -Path $backupFolder -ItemType Directory -Force | Out-Null
    }

    # Handle wildcards: copy all matching files
    Get-ChildItem -Path $prodPattern -File -ErrorAction SilentlyContinue | ForEach-Object {
        Write-Host "   Backing up $($_.FullName) -> $backupFolder" -ForegroundColor Green
        Copy-Item $_.FullName $backupFolder -Force
    }
}

# ===================================
# Stop Services & IIS
# ===================================
Write-Host "Stopping services..."

$Services = @(
    'fxc6.faxagent',
    'fxc6.dispatchagent',
    'fxc6.renderagent',
    'fxc6.filegateway',
    'fxc6.reportapi',
    'fxc6.smtpgatewayagent',
    'fxc6.smtplisteneragent',
    'fxc6.ServicesAPI',
    'MSSQL$FXCDB'
)

foreach ($Name in $Services) {
    Write-Host "   Stopping $Name ..." -ForegroundColor Green

    $svc = Get-Service -Name $Name -ErrorAction SilentlyContinue
    if (-not $svc) {
        Write-Host "   Service '$Name' not active" -ForegroundColor DarkGray
        continue
    }
        if ($svc.Status -eq 'Stopped') {
        Write-Host "   Already stopped" -ForegroundColor DarkGray
    } else {
        Write-Host "   Stopping service: $Name" -ForegroundColor DarkGray
        Stop-Service -name $Name -Force -ErrorAction SilentlyContinue
        Start-Sleep -Seconds 10
    }

    $cim = Get-CimInstance Win32_Service -Filter "Name='$Name'" -ErrorAction Stop
    if ($cim.ProcessId -gt 0) {
        Write-Host "   Stopping Process: $Name" -ForegroundColor DarkGray
        Stop-Process -Id $cim.ProcessId -Force -ErrorAction SilentlyContinue
    }
}

Write-Host "   Stopping IIS ..." -ForegroundColor Green
iisreset /STOP | Out-Null
Start-Sleep -Seconds 10

# ===================================
# Install New Files
# ===================================
Write-Host "Expanding .zip files"

if (Test-Path "$BaseReportDir\FaxCoreCode\FaxCoreWeb.zip") {
    Write-Host "   Expanding Web to $FCPathWeb"  -ForegroundColor Green
    Expand-Archive "$BaseReportDir\FaxCoreCode\FaxCoreWeb.zip" $FCPathWeb -Force
} else {
    Write-Error "FaxCoreWeb.zip missing"
}
if (Test-Path "$BaseReportDir\FaxCoreCode\FaxCoreSrc.zip") {
    if ($PatchType -eq "Base") {   # if doing a Web patch, this is not needed.
        Write-Host "   Expanding Svc to $FCPath"  -ForegroundColor Green
        Expand-Archive "$BaseReportDir\FaxCoreCode\FaxCoreSrc.zip" $FCPath -Force
    }
} else {
    Write-Error "FaxCoreSrc.zip missing"
}

# ===================================
# XML Helpers (Schema-Aware Phase 1)
# ===================================
function Test-XmlXPath {
    param (
        [Parameter(Mandatory)][xml]$Xml,
        [Parameter(Mandatory)][string]$XPath
    )
    try {
        return $null -ne $Xml.SelectSingleNode($XPath)
    } catch {
        return $false
    }
}

function Ensure-XmlPath {
    param (
        [Parameter(Mandatory)][xml]$Xml,
        [Parameter(Mandatory)][string]$XPath
    )

    $current = $Xml.DocumentElement
    foreach ($part in $XPath.Trim('/').Split('/') | Select-Object -Skip 1) {
        $next = $current.SelectSingleNode($part)
        if (-not $next) {
            $next = $Xml.CreateElement($part)
            $current.AppendChild($next) | Out-Null
        }
        $current = $next
    }
    return $current
}

function Merge-XmlNode {
    param (
        [xml]$SourceXml,
        [xml]$TargetXml,
        [string]$NodePath
    )

    $fullPath = if ($NodePath.StartsWith('/')) { $NodePath } else { "/configuration/$NodePath" }
    $srcNode  = $SourceXml.SelectSingleNode($fullPath)
    if (-not $srcNode) { return }

    $parentPath = $fullPath.Substring(0, $fullPath.LastIndexOf('/'))
    $parentNode = $TargetXml.SelectSingleNode($parentPath)
    if (-not $parentNode) { return }

    $existing = $null

    foreach ($child in $parentNode.ChildNodes) {
        if ($child.Name -ne $srcNode.Name) { continue }

        # If no attributes, first match wins (original behavior)
        if ($srcNode.Attributes.Count -eq 0) {
            $existing = $child
            break
        }

        $allMatch = $true
        foreach ($attr in $srcNode.Attributes) {
            if ($child.GetAttribute($attr.Name) -ne $attr.Value) {
                $allMatch = $false
                break
            }
        }

        if ($allMatch) {
            $existing = $child
            break
        }
    }

    $imported = $TargetXml.ImportNode($srcNode, $true)

    if ($existing) {
        $parentNode.ReplaceChild($imported, $existing) | Out-Null
    } else {
        $parentNode.AppendChild($imported) | Out-Null
    }
}


function Merge-ConfigFile {
    param (
        [Parameter(Mandatory)][string]$ProdPath,
        [Parameter(Mandatory)][string]$ConfigFile,
        [string[]]$NodesToCopy = @(),
        [string[]]$KeysToCopy  = @()
    )

    Write-Host "Updating $ConfigFile"

    $prodConfig   = Join-Path $ProdPath $ConfigFile
    $backupConfig = "$prodConfig.bak"
    $defltConfig  = Join-Path (Join-Path $ProdPath 'defaultConfigs') $ConfigFile

    if (Test-Path -Path $prodConfig) {
        Copy-Item $prodConfig $backupConfig -Force
        if ((Test-Path -Path $defltConfig)) {
            Copy-Item $defltConfig $prodConfig -Force
        }

        [xml]$newXml = Get-Content $prodConfig
        [xml]$oldXml = Get-Content $backupConfig

        foreach ($key in $KeysToCopy) {
            $keyXPath = "/configuration/appSettings/add[@key='$key']"
            if ((Test-XmlXPath -Xml $oldXml -XPath $keyXPath)) {
                Write-Host "   Updating key: $key" -ForegroundColor Green
                $srcKey      = $oldXml.SelectSingleNode($keyXPath)
                $appSettings = Ensure-XmlPath -Xml $newXml -XPath "/configuration/appSettings"
                $existing    = $appSettings.SelectSingleNode("add[@key='$key']")
                $imported    = $newXml.ImportNode($srcKey, $true)

                if ($existing) { $appSettings.ReplaceChild($imported, $existing) | Out-Null }
                else { $appSettings.AppendChild($imported) | Out-Null }
            } else {
               Write-Host "   Key not found or invalid, skipping: $key" -ForegroundColor Yellow
            }
        }

        foreach ($nodePath in $NodesToCopy) {
            $fullPath = if ($nodePath.StartsWith('/')) { $nodePath } else { "/configuration/$nodePath" }
            if ($oldXml.SelectSingleNode($fullPath)) {
                Write-Host "   Updating node: $fullPath" -ForegroundColor Green
                Merge-XmlNode -SourceXml $oldXml -TargetXml $newXml -NodePath $nodePath
            } else {
                Write-Host "   Source node not found, skipping: $fullPath" -ForegroundColor Yellow
            }
        }

        $newXml.Save($prodConfig)
        Remove-Item $backupConfig -Force
    } else {
        Write-Host "   Skipping updating $ProdConfig because it does not exist" -ForegroundColor Green
    }
}

# ===================================
# Apply Config Merges
# ===================================
Write-Host "Patch .config files"

Merge-ConfigFile `
    -ProdPath "$FCPathWeb" `
    -ConfigFile "web.config" `
    -KeysToCopy @(
        "web:title",
        "2fa:IsEnable",
        "2fa:appName",
        "accactivation:enable",
        "mail:support",
        "mail:billing",
        "mail:info",
        "mvPrivacy",
        "usecdn",
        "printerapp:folder",
        "registration:enable",
        "migration:databasename",
        "migration:databasecompatibilitylevel",
        "accountlockmaxtries",
        "accountlockmax:minutes"
    ) `
    -NodesToCopy @(
        "/configuration/DbConnectionSetting",
        "/configuration/smtpSection",
        "/configuration/oktaOAuth",
        "/configuration/googleOAuth",
        "/configuration/facebookOAuth",
        "/configuration/linkedInOAuth",
        "/configuration/twitterOAuth",
        "/configuration/azureAdOAuth",
        "/configuration/domainSection",
        "/configuration/cspDomain",
        "/configuration/passwordPolicy",
        "/configuration/saml2Config",
        "/configuration/sustainsys.saml2"
    )

If ($PatchType -eq "Base") {  # Exclude this if doing just a web patch
    Merge-ConfigFile `
        -ProdPath "$FCPath\svc.Dispatcher" `
        -ConfigFile "DispatchAgent.exe.config" `
        -KeysToCopy @(
            "EnableEmailSenderName",
            "PurgeInterval",
            "PurgeDebug",
            "PurgeLogStartTime",
            "PurgeDisable"
        )

    Merge-ConfigFile `
        -ProdPath "$FCPath\svc.Faxagent" `
        -ConfigFile "Faxagent.exe.config" `
        -KeysToCopy @(
            "AgentID",
            "DriverType"
        ) `
        -NodesToCopy @(
            "/configuration/Virtual/EtherFaxDR",
            "/configuration/Virtual/EtherFax"
        )

    Merge-ConfigFile `
        -ProdPath "$FCPath\svc.InternalAPI" `
        -ConfigFile "FaxCore.Internal.Api.exe.config" `
        -NodesToCopy @(
            "/configuration/DbConnectionSetting"
        )

    Merge-ConfigFile `
        -ProdPath "$FCPath\svc.RenderAgent" `
        -ConfigFile "RenderAgent.exe.config" `
        -KeysToCopy @(
            "PDFRenderingEngine",
            "XLSRenderingEngine",
            "DOCRenderingEngine"
        )

    Merge-ConfigFile `
        -ProdPath "$FCPath\svc.Report" `
        -ConfigFile "FaxCore.Report.Api.exe.config" `
        -KeysToCopy @(
            "NativeCountryCode"
        ) `
        -NodesToCopy @(
            "/configuration/DbConnectionSetting"
        )

    Merge-ConfigFile `
        -ProdPath "$FCPath\svc.SMTPGateway" `
        -ConfigFile "SMTPGateway.exe.config" `
        -NodesToCopy @(
            "/configuration/SMTPGateway/SMTP"
        )

    Merge-ConfigFile `
        -ProdPath "$FCPath\svc.SMTPListenerAgent" `
        -ConfigFile "SMTPListenerAgent.exe.config" `
        -NodesToCopy @(
            "/configuration/SMTPListener"
        )
}

# ===================================
# Restore Branding
# ===================================

Write-Host "Restoring branding..."

foreach ($file in $brandingFiles) {
    $backupPattern = Join-Path (Join-Path $BackupRoot $webFolder) $file
    $prodFolder    = Join-Path $FCPathWeb ([IO.Path]::GetDirectoryName($file))

    # Get all files matching the wildcard in backup
    Get-ChildItem -Path $backupPattern -File -ErrorAction SilentlyContinue | ForEach-Object {
        Write-Host "   Restoring $($_.FullName) -> $prodFolder" -ForegroundColor Green
        Copy-Item $_.FullName $prodFolder -Force
    }
}

# ===================================
# Start IIS & Finish
# ===================================
Write-Host "Restarting IIS"
iisreset /START | Out-Null
Start-Sleep 5
Start-Process "http://localhost/admin"

Write-Host "Wait for login page (Which is updateing the SQL FXC DB), then press ENTER..." -ForegroundColor Red

Read-Host
Write-Host "`nNow Restarting Faxcore." -ForegroundColor Green
& "$BaseReportDir\ServicesRestartEv6.bat"

Copy-Item $LogFile $BackupRoot -Force
Write-Host "`nMPatch v3.1 completed successfully!" -ForegroundColor Green
Write-Host "Backup: $BackupRoot"
Write-Host "Log: $LogFile"

Stop-Transcript
